<?php
/**
 * Page/post approval functionality.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * Page/post approval functionality.
 *
 * Handles the approval of duplicate pages/posts to replace originals.
 *
 * @since 1.2.0
 */
class Respira_Approver {

	/**
	 * Approve a duplicate page/post to replace the original.
	 *
	 * @since 1.2.0
	 * @param int $duplicate_id The ID of the duplicate post to approve.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function approve_duplicate( $duplicate_id ) {
		// Verify it's a duplicate.
		if ( ! Respira_Duplicator::is_duplicate( $duplicate_id ) ) {
			return new WP_Error(
				'respira_not_duplicate',
				__( 'This page/post is not a Respira duplicate and cannot be approved.', 'respira-for-wordpress' ),
				array( 'status' => 400 )
			);
		}

		// Get the original post ID.
		$original_id = Respira_Duplicator::get_original_id( $duplicate_id );
		if ( ! $original_id ) {
			return new WP_Error(
				'respira_original_not_found',
				__( 'Original post not found for this duplicate.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		// Get post objects.
		$duplicate = get_post( $duplicate_id );
		$original  = get_post( $original_id );

		if ( ! $duplicate || ! $original ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		// Get original slug.
		$original_slug = $original->post_name;
		if ( empty( $original_slug ) ) {
			$original_slug = sanitize_title( $original->post_title );
		}

		// Handle original page: unpublish and rename slug.
		$old_slug = self::handle_slug_conflict( $original_slug . '-old', $original_id );
		$result   = wp_update_post(
			array(
				'ID'          => $original_id,
				'post_status' => 'draft',
				'post_name'   => $old_slug,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Update duplicate: set to original slug and publish.
		$result = wp_update_post(
			array(
				'ID'          => $duplicate_id,
				'post_status' => 'publish',
				'post_name'   => $original_slug,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Remove duplicate marker from approved page.
		delete_post_meta( $duplicate_id, '_respira_duplicate' );
		delete_post_meta( $duplicate_id, '_respira_original_id' );
		delete_post_meta( $duplicate_id, '_respira_duplicated_at' );

		// Store approval metadata.
		update_post_meta( $duplicate_id, '_respira_approved_at', current_time( 'mysql' ) );
		update_post_meta( $duplicate_id, '_respira_approved_by', get_current_user_id() );
		update_post_meta( $duplicate_id, '_respira_old_post_id', $original_id );

		// Log approval action.
		if ( class_exists( 'Respira_Auth' ) ) {
			Respira_Auth::log_action(
				null,
				get_current_user_id(),
				'approve_duplicate',
				$duplicate->post_type,
				$duplicate_id,
				array(
					'original_id' => $original_id,
					'old_slug'    => $old_slug,
					'new_slug'    => $original_slug,
				)
			);
		}

		/**
		 * Fires after a duplicate has been approved.
		 *
		 * @since 1.2.0
		 * @param int $duplicate_id The ID of the approved duplicate.
		 * @param int $original_id  The ID of the original post (now unpublished).
		 */
		do_action( 'respira_after_approve_duplicate', $duplicate_id, $original_id );

		return true;
	}

	/**
	 * Reject a duplicate page/post (delete it).
	 *
	 * @since 1.4.2
	 * @param int $duplicate_id The ID of the duplicate post to reject.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function reject_duplicate( $duplicate_id ) {
		// Verify it's a duplicate.
		if ( ! Respira_Duplicator::is_duplicate( $duplicate_id ) ) {
			return new WP_Error(
				'respira_not_duplicate',
				__( 'This page/post is not a Respira duplicate and cannot be rejected.', 'respira-for-wordpress' ),
				array( 'status' => 400 )
			);
		}

		// Get the original post ID for logging.
		$original_id = Respira_Duplicator::get_original_id( $duplicate_id );
		
		// Get post object before deletion.
		$duplicate = get_post( $duplicate_id );
		if ( ! $duplicate ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		// Delete the duplicate permanently.
		$result = wp_delete_post( $duplicate_id, true );

		if ( ! $result ) {
			return new WP_Error(
				'respira_delete_failed',
				__( 'Failed to delete the duplicate.', 'respira-for-wordpress' ),
				array( 'status' => 500 )
			);
		}

		// Log rejection action.
		if ( class_exists( 'Respira_Auth' ) ) {
			Respira_Auth::log_action(
				null,
				get_current_user_id(),
				'reject_duplicate',
				$duplicate->post_type,
				$duplicate_id,
				array(
					'original_id' => $original_id,
					'title'       => $duplicate->post_title,
				)
			);
		}

		/**
		 * Fires after a duplicate has been rejected.
		 *
		 * @since 1.4.2
		 * @param int $duplicate_id The ID of the rejected duplicate.
		 * @param int $original_id  The ID of the original post.
		 */
		do_action( 'respira_after_reject_duplicate', $duplicate_id, $original_id );

		return true;
	}

	/**
	 * Handle slug conflicts by appending numbers.
	 *
	 * @since 1.2.0
	 * @param string $desired_slug The desired slug.
	 * @param int    $post_id     The post ID to exclude from conflict check.
	 * @return string A unique slug.
	 */
	public static function handle_slug_conflict( $desired_slug, $post_id = 0 ) {
		global $wpdb;

		$slug = $desired_slug;
		$num  = 2;

		// Check if slug exists (excluding current post).
		$check_sql = "SELECT post_name FROM $wpdb->posts WHERE post_name = %s AND ID != %d LIMIT 1";
		$exists    = $wpdb->get_var( $wpdb->prepare( $check_sql, $slug, $post_id ) );

		// If slug exists, append number until unique.
		while ( $exists ) {
			$slug = $desired_slug . '-' . $num;
			$exists = $wpdb->get_var( $wpdb->prepare( $check_sql, $slug, $post_id ) );
			$num++;
		}

		return $slug;
	}

	/**
	 * Get error data for API responses when trying to edit original.
	 *
	 * @since 1.2.0
	 * @param int      $original_id  The original post ID.
	 * @param int|null $duplicate_id Optional. The duplicate post ID if one exists.
	 * @return array Error data array with instructions and links.
	 */
	public static function get_approval_error_data( $original_id, $duplicate_id = null ) {
		$post = get_post( $original_id );
		if ( ! $post ) {
			return array();
		}

		$post_type_label = 'page' === $post->post_type ? __( 'page', 'respira-for-wordpress' ) : __( 'post', 'respira-for-wordpress' );
		$rest_namespace  = RESPIRA_REST_NAMESPACE;

		// Check if duplicate exists if not provided.
		if ( null === $duplicate_id ) {
			$duplicates = Respira_Duplicator::get_duplicates( $original_id );
			if ( ! empty( $duplicates ) ) {
				$duplicate_id = $duplicates[0]->ID;
			}
		}

		$error_data = array(
			'status'       => 403,
			'instructions' => array(),
			'links'        => array(
				'approvals_page'  => admin_url( 'admin.php?page=respira-approvals' ),
				'settings_page'   => admin_url( 'admin.php?page=respira-settings' ),
				'duplicate_endpoint' => rest_url( $rest_namespace . '/' . $post->post_type . 's/' . $original_id . '/duplicate' ),
			),
			'original_id'  => $original_id,
			'duplicate_id' => $duplicate_id,
		);

		if ( $duplicate_id ) {
			// Duplicate exists.
			$error_data['instructions'] = array(
				'step1' => sprintf(
					/* translators: 1: post type label, 2: duplicate ID */
					__( 'A duplicate %1$s already exists (ID: %2$d). You can edit it directly.', 'respira-for-wordpress' ),
					$post_type_label,
					$duplicate_id
				),
				'step2' => sprintf(
					/* translators: 1: post type label */
					__( 'Edit the duplicate %1$s: PUT /wp-json/respira/v1/%2$ss/%3$d', 'respira-for-wordpress' ),
					$post_type_label,
					$post->post_type,
					$duplicate_id
				),
				'step3' => __( 'Approve the duplicate in WordPress admin to replace the original.', 'respira-for-wordpress' ),
			);
			$error_data['links']['edit_duplicate'] = rest_url( $rest_namespace . '/' . $post->post_type . 's/' . $duplicate_id );
		} else {
			// No duplicate exists.
			$error_data['instructions'] = array(
				'step1' => sprintf(
					/* translators: 1: post type label */
					__( 'Create a duplicate %1$s: POST /wp-json/respira/v1/%2$ss/%3$d/duplicate', 'respira-for-wordpress' ),
					$post_type_label,
					$post->post_type,
					$original_id
				),
				'step2' => sprintf(
					/* translators: 1: post type label */
					__( 'Edit the duplicate %1$s instead of the original.', 'respira-for-wordpress' ),
					$post_type_label
				),
				'step3' => __( 'Approve the duplicate in WordPress admin to replace the original.', 'respira-for-wordpress' ),
			);
		}

		// Add setting-related instruction if direct editing is disabled.
		$allow_direct_edit = get_option( 'respira_allow_direct_edit', 0 );
		if ( ! $allow_direct_edit ) {
			$error_data['instructions']['setting_note'] = __( 'Note: Direct editing of original pages is disabled in settings. Enable it in Respira → Settings if you want to use force=true parameter.', 'respira-for-wordpress' );
		} else {
			$error_data['instructions']['force_note'] = __( 'Alternatively, you can use force=true parameter if direct editing is enabled in settings.', 'respira-for-wordpress' );
		}

		return $error_data;
	}
}

