<?php
/**
 * Brizy element schema generator.
 *
 * Generates schemas for all registered elements with property definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/brizy-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Brizy element schema class.
 *
 * @since 1.4.0
 */
class Respira_Brizy_Element_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		$version = defined( 'BRIZY_VERSION' ) ? BRIZY_VERSION : 'unknown';
		parent::__construct( 'brizy', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.4.0
	 * @param  array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		$schema = array(
			'builder'          => 'brizy',
			'builder_version'  => $this->builder_version,
			'available_elements' => array(),
			'quick_reference'  => $this->get_quick_reference(),
			'structure_notes'  => $this->get_structure_notes(),
		);

		// Get schemas for elements used on the page (or all elements if none specified).
		if ( empty( $elements_used ) ) {
			$all_elements = Respira_Brizy_Element_Registry::get_all_elements();
			$elements_used = array_column( $all_elements, 'name' );
		}

		foreach ( $elements_used as $element_name ) {
			$element = Respira_Brizy_Element_Registry::get_element( $element_name );
			if ( ! $element ) {
				continue;
			}

			$element_schema = array(
				'title'       => $element['title'],
				'description' => $element['description'],
				'type'        => $element['type'],
				'properties'  => array(),
			);

			// Generate property schemas.
			if ( ! empty( $element['properties'] ) ) {
				foreach ( $element['properties'] as $property ) {
					$element_schema['properties'][ $property ] = $this->get_property_schema( $property, $element['type'] );
				}
			}

			$schema['available_elements'][ $element_name ] = $element_schema;
		}

		return $schema;
	}

	/**
	 * Get property schema for an element property.
	 *
	 * @since  1.4.0
	 * @param  string $property     Property name.
	 * @param  string $element_type Element type.
	 * @return array Property schema.
	 */
	private function get_property_schema( $property, $element_type ) {
		// Map common properties to their types and formats.
		$property_map = array(
			// Text properties.
			'text'       => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<p>Sample text</p>',
			),
			'typography' => array(
				'type'    => 'object',
				'format'  => 'typography',
				'example' => array(
					'fontFamily' => 'Arial',
					'fontSize'   => 16,
					'fontWeight' => 400,
				),
			),
			'color'      => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#000000',
			),
			'colorHex'   => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#000000',
			),
			'bgColorHex' => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#FFFFFF',
			),
			'borderColorHex' => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#CCCCCC',
			),

			// Image properties.
			'imageSrc'    => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/image.jpg',
			),
			'imageWidth'  => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 800,
			),
			'imageHeight' => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 600,
			),
			'alt'         => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Image description',
			),

			// Link properties.
			'link'      => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),
			'linkType'  => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'external',
				'options' => array( 'external', 'anchor', 'popup' ),
			),

			// Icon properties.
			'name'      => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'icon-name',
			),
			'size'      => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 32,
			),

			// Spacing properties.
			'height'    => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 50,
			),
			'width'     => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 100,
			),
			'borderWidth' => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 1,
			),

			// Map properties.
			'address'   => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '123 Main St, City, State',
			),
			'zoom'      => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 12,
			),

			// Video properties.
			'video'      => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://youtube.com/watch?v=abc123',
			),
			'coverImageSrc' => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/cover.jpg',
			),
			'ratio'      => array(
				'type'    => 'string',
				'format'  => 'ratio',
				'example' => '16:9',
			),

			// Counter properties.
			'start'     => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 0,
			),
			'end'       => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 100,
			),
			'duration'  => array(
				'type'    => 'number',
				'format'  => 'milliseconds',
				'example' => 2000,
			),

			// Date properties.
			'date'      => array(
				'type'    => 'string',
				'format'  => 'date',
				'example' => '2024-12-31',
			),
			'timezone'  => array(
				'type'    => 'string',
				'format'  => 'timezone',
				'example' => 'America/New_York',
			),

			// Progress properties.
			'percentage' => array(
				'type'    => 'number',
				'format'  => 'percentage',
				'example' => 75,
			),
			'showPercentage' => array(
				'type'    => 'boolean',
				'format'  => 'boolean',
				'example' => true,
			),
			'barColorHex' => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#0000FF',
			),

			// Container properties.
			'tabs'      => array(
				'type'    => 'array',
				'format'  => 'tabs',
				'example' => array(),
			),
			'items'     => array(
				'type'    => 'array',
				'format'  => 'items',
				'example' => array(),
			),
			'columns'   => array(
				'type'    => 'array',
				'format'  => 'columns',
				'example' => array(),
			),

			// Gallery properties.
			'gridColumn' => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 3,
			),

			// Menu properties.
			'menuSelected' => array(
				'type'    => 'string',
				'format'  => 'menu_id',
				'example' => 'primary',
			),

			// Form properties.
			'fields'     => array(
				'type'    => 'array',
				'format'  => 'form_fields',
				'example' => array(),
			),
			'submitText' => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Submit',
			),

			// Code properties.
			'html'      => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<div>Custom HTML</div>',
			),

			// Generic properties.
			'url'       => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),
		);

		// Return the schema for this property, or a generic schema if not found.
		if ( isset( $property_map[ $property ] ) ) {
			return $property_map[ $property ];
		}

		// Generic property schema.
		return array(
			'type'    => 'string',
			'format'  => 'text',
			'example' => 'value',
		);
	}

	/**
	 * Get structure notes for Brizy.
	 *
	 * @since  1.4.0
	 * @return array Structure notes.
	 */
	private function get_structure_notes() {
		return array(
			'hierarchy'   => 'Section → Row → Column → Elements',
			'json_format' => 'Brizy uses JSON-based element structure stored in postmeta',
			'elements'    => 'Each element has type and value properties',
			'nesting'     => 'Container elements (Section, Row, Column) can contain child elements via items/columns arrays',
			'note'        => 'Always preserve the element hierarchy when modifying content',
		);
	}
}
