<?php
/**
 * Bricks Builder element registry.
 *
 * Dynamic element detection from Bricks Builder's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/bricks-intelligence
 * @since      1.4.0
 */

/**
 * Bricks Builder element registry class.
 *
 * @since 1.4.0
 */
class Respira_Bricks_Element_Registry {

	/**
	 * Get all registered elements with metadata.
	 *
	 * @since  1.4.0
	 * @return array Array of element information.
	 */
	public static function get_all_elements() {
		// Check cache first.
		$cached = get_transient( 'respira_bricks_elements' );
		if ( false !== $cached ) {
			return $cached;
		}

		$element_list = array();

		// Try to detect elements from Bricks Builder's registry.
		if ( class_exists( 'Bricks\Elements' ) ) {
			$element_list = self::detect_from_api();
		}

		// Fallback: use known element types from Bricks Builder documentation.
		if ( empty( $element_list ) ) {
			$element_list = self::get_known_elements();
		}

		// Cache for 24 hours.
		set_transient( 'respira_bricks_elements', $element_list, DAY_IN_SECONDS );

		return $element_list;
	}

	/**
	 * Detect elements from Bricks Builder API (if available).
	 *
	 * @since  1.4.0
	 * @return array Array of detected elements.
	 */
	private static function detect_from_api() {
		$elements = array();

		try {
			// Bricks provides access to elements via the Elements class.
			if ( class_exists( 'Bricks\Elements' ) ) {
				$bricks_elements = \Bricks\Elements::$elements;

				foreach ( $bricks_elements as $element_name => $element_class ) {
					$elements[] = array(
						'name'        => $element_name,
						'title'       => ucwords( str_replace( '-', ' ', $element_name ) ),
						'type'        => $element_name,
						'description' => '',
						'properties'  => array(),
					);
				}
			}
		} catch ( Exception $e ) {
			// Silently fall back to known elements.
		}

		return $elements;
	}

	/**
	 * Get known Bricks Builder elements from documentation.
	 *
	 * @since  1.4.0
	 * @return array Array of known elements.
	 */
	private static function get_known_elements() {
		return array(
			// Layout elements.
			array(
				'name'        => 'Container',
				'title'       => 'Container',
				'type'        => 'container',
				'description' => 'Container element for grouping other elements',
				'category'    => 'layout',
				'properties'  => array( 'tag', '_width', '_padding', '_margin', '_background', '_border' ),
			),
			array(
				'name'        => 'Block',
				'title'       => 'Block',
				'type'        => 'block',
				'description' => 'Div block element',
				'category'    => 'layout',
				'properties'  => array( 'tag', '_width', '_height', '_background', '_border' ),
			),
			array(
				'name'        => 'Section',
				'title'       => 'Section',
				'type'        => 'section',
				'description' => 'Section container',
				'category'    => 'layout',
				'properties'  => array( 'tag', '_width', '_padding', '_background' ),
			),
			array(
				'name'        => 'Div',
				'title'       => 'Div',
				'type'        => 'div',
				'description' => 'Generic div element',
				'category'    => 'layout',
				'properties'  => array( 'tag', '_width', '_height' ),
			),

			// Content elements.
			array(
				'name'        => 'Heading',
				'title'       => 'Heading',
				'type'        => 'heading',
				'description' => 'Heading element (H1-H6)',
				'category'    => 'content',
				'properties'  => array( 'text', 'tag', 'link', '_typography', '_color' ),
			),
			array(
				'name'        => 'Text',
				'title'       => 'Text',
				'type'        => 'text',
				'description' => 'Text element with rich content',
				'category'    => 'content',
				'properties'  => array( 'text', '_typography', '_color' ),
			),
			array(
				'name'        => 'Text Basic',
				'title'       => 'Text Basic',
				'type'        => 'text-basic',
				'description' => 'Basic text without editor',
				'category'    => 'content',
				'properties'  => array( 'text', '_typography', '_color' ),
			),
			array(
				'name'        => 'Button',
				'title'       => 'Button',
				'type'        => 'button',
				'description' => 'Button element',
				'category'    => 'content',
				'properties'  => array( 'text', 'link', 'type', 'icon', '_background', '_typography', '_border' ),
			),
			array(
				'name'        => 'Icon',
				'title'       => 'Icon',
				'type'        => 'icon',
				'description' => 'Icon element',
				'category'    => 'content',
				'properties'  => array( 'icon', 'link', '_color', '_fontSize' ),
			),
			array(
				'name'        => 'Icon Box',
				'title'       => 'Icon Box',
				'type'        => 'icon-box',
				'description' => 'Icon with title and description',
				'category'    => 'content',
				'properties'  => array( 'icon', 'title', 'description', 'link' ),
			),
			array(
				'name'        => 'Divider',
				'title'       => 'Divider',
				'type'        => 'divider',
				'description' => 'Horizontal divider line',
				'category'    => 'content',
				'properties'  => array( 'style', 'width', 'height', '_border' ),
			),
			array(
				'name'        => 'List',
				'title'       => 'List',
				'type'        => 'list',
				'description' => 'List element',
				'category'    => 'content',
				'properties'  => array( 'items', 'icon', '_typography' ),
			),
			array(
				'name'        => 'Code',
				'title'       => 'Code',
				'type'        => 'code',
				'description' => 'Code block with syntax highlighting',
				'category'    => 'content',
				'properties'  => array( 'code', 'language', 'lineNumbers' ),
			),
			array(
				'name'        => 'HTML',
				'title'       => 'HTML',
				'type'        => 'html',
				'description' => 'Custom HTML code',
				'category'    => 'content',
				'properties'  => array( 'html' ),
			),

			// Media elements.
			array(
				'name'        => 'Image',
				'title'       => 'Image',
				'type'        => 'image',
				'description' => 'Image element',
				'category'    => 'media',
				'properties'  => array( 'image', 'size', 'link', 'alt', 'objectFit', '_aspectRatio' ),
			),
			array(
				'name'        => 'Video',
				'title'       => 'Video',
				'type'        => 'video',
				'description' => 'Video element (YouTube, Vimeo, self-hosted)',
				'category'    => 'media',
				'properties'  => array( 'videoType', 'videoUrl', 'videoId', 'controls', 'autoplay', 'loop' ),
			),
			array(
				'name'        => 'Audio',
				'title'       => 'Audio',
				'type'        => 'audio',
				'description' => 'Audio player',
				'category'    => 'media',
				'properties'  => array( 'audioFile', 'autoplay', 'loop' ),
			),
			array(
				'name'        => 'Slider',
				'title'       => 'Slider',
				'type'        => 'slider',
				'description' => 'Content slider/carousel',
				'category'    => 'media',
				'properties'  => array( 'items', 'autoplay', 'speed', 'navigation', 'pagination' ),
			),
			array(
				'name'        => 'Carousel',
				'title'       => 'Carousel',
				'type'        => 'carousel',
				'description' => 'Image carousel',
				'category'    => 'media',
				'properties'  => array( 'images', 'autoplay', 'speed', 'slidesPerView' ),
			),
			array(
				'name'        => 'SVG',
				'title'       => 'SVG',
				'type'        => 'svg',
				'description' => 'SVG element',
				'category'    => 'media',
				'properties'  => array( 'svg', 'width', 'height', '_color' ),
			),

			// Interactive elements.
			array(
				'name'        => 'Accordion',
				'title'       => 'Accordion',
				'type'        => 'accordion',
				'description' => 'Accordion/collapsible content',
				'category'    => 'interactive',
				'properties'  => array( 'items', 'multipleOpen', 'firstOpen' ),
			),
			array(
				'name'        => 'Tabs',
				'title'       => 'Tabs',
				'type'        => 'tabs',
				'description' => 'Tabbed content',
				'category'    => 'interactive',
				'properties'  => array( 'items', 'layout', 'activeTab' ),
			),
			array(
				'name'        => 'Toggle',
				'title'       => 'Toggle',
				'type'        => 'toggle',
				'description' => 'Toggle content',
				'category'    => 'interactive',
				'properties'  => array( 'title', 'content', 'open' ),
			),
			array(
				'name'        => 'Modal',
				'title'       => 'Modal',
				'type'        => 'modal',
				'description' => 'Modal popup',
				'category'    => 'interactive',
				'properties'  => array( 'trigger', 'content', 'width', '_background' ),
			),
			array(
				'name'        => 'Popup',
				'title'       => 'Popup',
				'type'        => 'popup',
				'description' => 'Popup element',
				'category'    => 'interactive',
				'properties'  => array( 'trigger', 'content', 'animation' ),
			),

			// Form elements.
			array(
				'name'        => 'Form',
				'title'       => 'Form',
				'type'        => 'form',
				'description' => 'Form builder',
				'category'    => 'forms',
				'properties'  => array( 'fields', 'submitText', 'action', 'method' ),
			),

			// WordPress elements.
			array(
				'name'        => 'Posts',
				'title'       => 'Posts',
				'type'        => 'posts',
				'description' => 'Posts query loop',
				'category'    => 'wordpress',
				'properties'  => array( 'query', 'layout', 'postsPerPage', 'orderBy', 'order' ),
			),
			array(
				'name'        => 'Nav Menu',
				'title'       => 'Nav Menu',
				'type'        => 'nav-menu',
				'description' => 'Navigation menu',
				'category'    => 'wordpress',
				'properties'  => array( 'menu', 'mobileMenu', 'mobileBreakpoint' ),
			),
			array(
				'name'        => 'Sidebar',
				'title'       => 'Sidebar',
				'type'        => 'sidebar',
				'description' => 'Widget area',
				'category'    => 'wordpress',
				'properties'  => array( 'sidebar' ),
			),

			// Advanced elements.
			array(
				'name'        => 'Map',
				'title'       => 'Map',
				'type'        => 'map',
				'description' => 'Google Maps',
				'category'    => 'advanced',
				'properties'  => array( 'address', 'zoom', 'height', 'markers' ),
			),
			array(
				'name'        => 'Counter',
				'title'       => 'Counter',
				'type'        => 'counter',
				'description' => 'Animated counter',
				'category'    => 'advanced',
				'properties'  => array( 'start', 'end', 'duration', 'prefix', 'suffix' ),
			),
			array(
				'name'        => 'Progress Bar',
				'title'       => 'Progress Bar',
				'type'        => 'progress-bar',
				'description' => 'Progress bar',
				'category'    => 'advanced',
				'properties'  => array( 'value', 'max', 'showValue', '_background' ),
			),
			array(
				'name'        => 'Testimonials',
				'title'       => 'Testimonials',
				'type'        => 'testimonials',
				'description' => 'Testimonials carousel',
				'category'    => 'advanced',
				'properties'  => array( 'items', 'autoplay', 'layout' ),
			),
			array(
				'name'        => 'Pricing Table',
				'title'       => 'Pricing Table',
				'type'        => 'pricing-table',
				'description' => 'Pricing table',
				'category'    => 'advanced',
				'properties'  => array( 'title', 'price', 'features', 'button', 'highlight' ),
			),
			array(
				'name'        => 'Social Icons',
				'title'       => 'Social Icons',
				'type'        => 'social-icons',
				'description' => 'Social media icons',
				'category'    => 'advanced',
				'properties'  => array( 'items', 'layout', '_color' ),
			),
		);
	}

	/**
	 * Get element by name or type.
	 *
	 * @since  1.4.0
	 * @param  string $element_name Element name or type.
	 * @return array|null Element information or null if not found.
	 */
	public static function get_element( $element_name ) {
		$elements = self::get_all_elements();

		foreach ( $elements as $element ) {
			if ( $element['name'] === $element_name || $element['type'] === $element_name ) {
				return $element;
			}
		}

		return null;
	}

	/**
	 * Clear element cache.
	 *
	 * @since 1.4.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_bricks_elements' );
	}
}
