<?php
/**
 * Beaver Builder validator.
 *
 * Validates Beaver Builder module structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/beaver-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Beaver Builder validator class.
 *
 * @since 1.4.0
 */
class Respira_Beaver_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.4.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Validate each row in the layout.
		foreach ( $content as $node_id => $node ) {
			if ( ! is_array( $node ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: node ID */
						__( 'Node %s must be an array.', 'respira-for-wordpress' ),
						$node_id
					)
				);
				continue;
			}

			// Validate node type.
			if ( ! isset( $node['type'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: node ID */
						__( 'Node %s must have a type.', 'respira-for-wordpress' ),
						$node_id
					)
				);
				continue;
			}

			// Validate based on node type.
			switch ( $node['type'] ) {
				case 'row':
					$this->validate_row( $node, $node_id );
					break;
				case 'column':
					$this->validate_column( $node, $node_id );
					break;
				case 'module':
					$this->validate_module( $node, $node_id );
					break;
				default:
					$this->add_error(
						sprintf(
							/* translators: 1: node type, 2: node ID */
							__( 'Unknown node type %1$s for node %2$s.', 'respira-for-wordpress' ),
							$node['type'],
							$node_id
						)
					);
			}
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate a row node.
	 *
	 * @since  1.4.0
	 * @param  array  $row     Row node data.
	 * @param  string $node_id Node ID.
	 */
	private function validate_row( $row, $node_id ) {
		// Row must have columns.
		if ( ! isset( $row['columns'] ) || ! is_array( $row['columns'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Row %s must have columns array.', 'respira-for-wordpress' ),
					$node_id
				)
			);
			return;
		}

		// Validate row settings if present.
		if ( isset( $row['settings'] ) && is_array( $row['settings'] ) ) {
			$this->validate_row_settings( $row['settings'], $node_id );
		}
	}

	/**
	 * Validate row settings.
	 *
	 * @since  1.4.0
	 * @param  array  $settings Row settings.
	 * @param  string $node_id  Node ID.
	 */
	private function validate_row_settings( $settings, $node_id ) {
		// Validate common row settings.
		if ( isset( $settings['bg_color'] ) && ! $this->validate_color( $settings['bg_color'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Invalid background color in row %s.', 'respira-for-wordpress' ),
					$node_id
				)
			);
		}

		if ( isset( $settings['bg_image_src'] ) && ! empty( $settings['bg_image_src'] ) && ! $this->validate_url( $settings['bg_image_src'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Invalid background image URL in row %s.', 'respira-for-wordpress' ),
					$node_id
				)
			);
		}
	}

	/**
	 * Validate a column node.
	 *
	 * @since  1.4.0
	 * @param  array  $column  Column node data.
	 * @param  string $node_id Node ID.
	 */
	private function validate_column( $column, $node_id ) {
		// Column must have a parent (row).
		if ( ! isset( $column['parent'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Column %s must have a parent row.', 'respira-for-wordpress' ),
					$node_id
				)
			);
		}

		// Validate column settings if present.
		if ( isset( $column['settings'] ) && is_array( $column['settings'] ) ) {
			$this->validate_column_settings( $column['settings'], $node_id );
		}
	}

	/**
	 * Validate column settings.
	 *
	 * @since  1.4.0
	 * @param  array  $settings Column settings.
	 * @param  string $node_id  Node ID.
	 */
	private function validate_column_settings( $settings, $node_id ) {
		// Validate column size/width.
		if ( isset( $settings['size'] ) ) {
			$valid_sizes = array( '100', '50', '33.33', '25', '66.66', '75', '20', '40', '60', '80' );
			if ( ! in_array( (string) $settings['size'], $valid_sizes, true ) ) {
				// Allow custom sizes, but warn in debug mode.
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					error_log( "Respira: Non-standard column size in column {$node_id}: {$settings['size']}" );
				}
			}
		}

		// Validate colors.
		if ( isset( $settings['bg_color'] ) && ! $this->validate_color( $settings['bg_color'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Invalid background color in column %s.', 'respira-for-wordpress' ),
					$node_id
				)
			);
		}
	}

	/**
	 * Validate a module node.
	 *
	 * @since  1.4.0
	 * @param  array  $module  Module node data.
	 * @param  string $node_id Node ID.
	 */
	private function validate_module( $module, $node_id ) {
		// Module must have a parent (column).
		if ( ! isset( $module['parent'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Module %s must have a parent column.', 'respira-for-wordpress' ),
					$node_id
				)
			);
		}

		// Module must have settings with type.
		if ( ! isset( $module['settings'] ) || ! is_array( $module['settings'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Module %s must have settings array.', 'respira-for-wordpress' ),
					$node_id
				)
			);
			return;
		}

		if ( ! isset( $module['settings']['type'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: node ID */
					__( 'Module %s must have a type in settings.', 'respira-for-wordpress' ),
					$node_id
				)
			);
			return;
		}

		// Validate module type is known.
		$module_type     = $module['settings']['type'];
		$known_module = Respira_Beaver_Module_Registry::get_module( $module_type );
		if ( ! $known_module ) {
			// Unknown module - allow it (may be from extension).
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( "Respira: Unknown Beaver Builder module type: {$module_type}" );
			}
		}

		// Validate module settings.
		$this->validate_module_settings( $module['settings'], $module_type, $node_id );
	}

	/**
	 * Validate module settings.
	 *
	 * @since  1.4.0
	 * @param  array  $settings    Module settings.
	 * @param  string $module_type Module type.
	 * @param  string $node_id     Node ID.
	 */
	private function validate_module_settings( $settings, $module_type, $node_id ) {
		// Validate common properties.
		foreach ( $settings as $prop_name => $prop_value ) {
			// Color validation.
			if ( strpos( $prop_name, 'color' ) !== false || strpos( $prop_name, '_color' ) !== false ) {
				if ( ! empty( $prop_value ) && ! $this->validate_color( $prop_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: property name, 2: module type, 3: node ID */
							__( 'Invalid color format for %1$s in %2$s module %3$s. Use hex format (#RRGGBB).', 'respira-for-wordpress' ),
							$prop_name,
							$module_type,
							$node_id
						)
					);
				}
			}

			// URL validation.
			if ( ( strpos( $prop_name, 'url' ) !== false || strpos( $prop_name, 'link' ) !== false || strpos( $prop_name, '_src' ) !== false ) && $prop_name !== 'link_target' ) {
				if ( ! empty( $prop_value ) && is_string( $prop_value ) && ! $this->validate_url( $prop_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: property name, 2: module type, 3: node ID */
							__( 'Invalid URL format for %1$s in %2$s module %3$s.', 'respira-for-wordpress' ),
							$prop_name,
							$module_type,
							$node_id
						)
					);
				}
			}

			// Numeric validation.
			if ( in_array( $prop_name, array( 'size', 'height', 'width', 'spacing', 'offset', 'speed', 'pause', 'number', 'max_number', 'posts_per_page' ), true ) ) {
				if ( ! empty( $prop_value ) && ! is_numeric( $prop_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: property name, 2: module type, 3: node ID */
							__( 'Invalid numeric value for %1$s in %2$s module %3$s.', 'respira-for-wordpress' ),
							$prop_name,
							$module_type,
							$node_id
						)
					);
				}
			}

			// Array validation for multi-value fields.
			if ( in_array( $prop_name, array( 'photos', 'items', 'tabs', 'features', 'icons', 'platforms' ), true ) ) {
				if ( ! empty( $prop_value ) && ! is_array( $prop_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: property name, 2: module type, 3: node ID */
							__( 'Property %1$s must be an array in %2$s module %3$s.', 'respira-for-wordpress' ),
							$prop_name,
							$module_type,
							$node_id
						)
					);
				}
			}
		}
	}

	/**
	 * Validate Beaver Builder data array structure.
	 *
	 * @since  1.4.0
	 * @param  mixed $data Data to validate.
	 * @return bool True if valid structure.
	 */
	public function validate_data_structure( $data ) {
		if ( ! is_array( $data ) ) {
			$this->add_error( __( 'Beaver Builder data must be an array.', 'respira-for-wordpress' ) );
			return false;
		}

		// Each item should be a node with specific structure.
		foreach ( $data as $node_id => $node ) {
			if ( ! is_array( $node ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: node ID */
						__( 'Node %s is not an array.', 'respira-for-wordpress' ),
						$node_id
					)
				);
				return false;
			}

			if ( ! isset( $node['type'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: node ID */
						__( 'Node %s missing type.', 'respira-for-wordpress' ),
						$node_id
					)
				);
				return false;
			}
		}

		return true;
	}
}
