<?php
/**
 * Beaver Builder module schema generator.
 *
 * Generates schemas for all registered modules with property definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/beaver-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Beaver Builder module schema class.
 *
 * @since 1.4.0
 */
class Respira_Beaver_Module_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		$version = defined( 'FL_BUILDER_VERSION' ) ? FL_BUILDER_VERSION : 'unknown';
		parent::__construct( 'beaver', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.4.0
	 * @param  array $modules_used Optional. Array of module names used on the page.
	 * @return array Builder schema with module information.
	 */
	public function get_builder_schema( $modules_used = array() ) {
		$schema = array(
			'builder'           => 'beaver',
			'builder_version'   => $this->builder_version,
			'available_modules' => array(),
			'quick_reference'   => $this->get_quick_reference(),
			'structure_notes'   => $this->get_structure_notes(),
		);

		// Get schemas for modules used on the page (or all modules if none specified).
		if ( empty( $modules_used ) ) {
			$all_modules   = Respira_Beaver_Module_Registry::get_all_modules();
			$modules_used = array_column( $all_modules, 'type' );
		}

		foreach ( $modules_used as $module_name ) {
			$module = Respira_Beaver_Module_Registry::get_module( $module_name );
			if ( ! $module ) {
				continue;
			}

			$module_schema = array(
				'title'       => $module['title'],
				'description' => $module['description'],
				'type'        => $module['type'],
				'category'    => $module['category'],
				'properties'  => array(),
			);

			// Generate property schemas.
			if ( ! empty( $module['properties'] ) ) {
				foreach ( $module['properties'] as $property ) {
					$module_schema['properties'][ $property ] = $this->get_property_schema( $property, $module['type'] );
				}
			}

			$schema['available_modules'][ $module['type'] ] = $module_schema;
		}

		return $schema;
	}

	/**
	 * Get property schema for a module property.
	 *
	 * @since  1.4.0
	 * @param  string $property    Property name.
	 * @param  string $module_type Module type.
	 * @return array Property schema.
	 */
	private function get_property_schema( $property, $module_type ) {
		// Map common properties to their types and formats.
		$property_map = array(
			// Text properties.
			'heading'     => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Heading Text',
			),
			'text'        => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<p>Rich text content</p>',
			),
			'html'        => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<div>Custom HTML</div>',
			),
			'title'       => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Title',
			),
			'caption'     => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Image caption',
			),

			// Link properties.
			'link'        => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),
			'link_url'    => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),
			'link_target' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => '_blank',
				'options' => array( '_self', '_blank' ),
			),
			'button_url'  => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),

			// Image/Media properties.
			'photo'       => array(
				'type'    => 'number',
				'format'  => 'attachment_id',
				'example' => 123,
			),
			'photo_src'   => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/image.jpg',
			),
			'photos'      => array(
				'type'    => 'array',
				'format'  => 'photo_array',
				'example' => array(),
			),
			'poster'      => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/poster.jpg',
			),

			// Video properties.
			'video_type'  => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'youtube',
				'options' => array( 'youtube', 'vimeo', 'embed', 'media_library' ),
			),
			'video'       => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://youtube.com/watch?v=abc123',
			),
			'video_embed' => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<iframe src="..."></iframe>',
			),

			// Audio properties.
			'audio'       => array(
				'type'    => 'number',
				'format'  => 'attachment_id',
				'example' => 123,
			),
			'audio_src'   => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/audio.mp3',
			),

			// Style properties.
			'tag'         => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'h2',
				'options' => array( 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'div' ),
			),
			'font_family' => array(
				'type'    => 'string',
				'format'  => 'font_family',
				'example' => 'Arial',
			),
			'font_size'   => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 24,
			),
			'color'       => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#000000',
			),
			'bg_color'    => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#FFFFFF',
			),
			'text_color'  => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#000000',
			),
			'border_color' => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#CCCCCC',
			),

			// Layout properties.
			'align'       => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'center',
				'options' => array( 'left', 'center', 'right' ),
			),
			'width'       => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'auto',
				'options' => array( 'auto', 'full', 'custom' ),
			),
			'size'        => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 50,
			),
			'height'      => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 400,
			),
			'spacing'     => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 20,
			),
			'photo_spacing' => array(
				'type'    => 'number',
				'format'  => 'pixels',
				'example' => 10,
			),

			// Button properties.
			'style'       => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'flat',
			),
			'button_text' => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Click Here',
			),

			// Icon properties.
			'icon'        => array(
				'type'    => 'string',
				'format'  => 'icon_class',
				'example' => 'fa fa-heart',
			),
			'icons'       => array(
				'type'    => 'array',
				'format'  => 'icon_array',
				'example' => array(),
			),

			// Gallery/Slider properties.
			'layout'      => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'grid',
			),
			'show_captions' => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),
			'click_action' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'lightbox',
				'options' => array( 'none', 'lightbox', 'link' ),
			),
			'transition'  => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'fade',
			),
			'speed'       => array(
				'type'    => 'number',
				'format'  => 'milliseconds',
				'example' => 1000,
			),
			'pause'       => array(
				'type'    => 'number',
				'format'  => 'milliseconds',
				'example' => 4000,
			),
			'auto_play'   => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),

			// Accordion/Tabs properties.
			'items'       => array(
				'type'    => 'array',
				'format'  => 'items_array',
				'example' => array(),
			),
			'tabs'        => array(
				'type'    => 'array',
				'format'  => 'tabs_array',
				'example' => array(),
			),
			'open_first'  => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),
			'collapse'    => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),
			'border_style' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'solid',
			),

			// Form properties.
			'name_toggle' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'show',
			),
			'email_toggle' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'show',
			),
			'phone_toggle' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'show',
			),
			'subject_toggle' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'show',
			),
			'message_toggle' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'show',
			),
			'success_action' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'message',
			),
			'success_message' => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Thank you for your message!',
			),

			// Pricing table properties.
			'price'       => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '$99',
			),
			'duration'    => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '/month',
			),
			'features'    => array(
				'type'    => 'array',
				'format'  => 'features_array',
				'example' => array(),
			),
			'highlight'   => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'no',
			),

			// Posts properties.
			'post_type'   => array(
				'type'    => 'string',
				'format'  => 'post_type',
				'example' => 'post',
			),
			'order'       => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'DESC',
				'options' => array( 'DESC', 'ASC' ),
			),
			'offset'      => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 0,
			),
			'posts_per_page' => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 10,
			),
			'post_taxonomies' => array(
				'type'    => 'array',
				'format'  => 'taxonomy_array',
				'example' => array(),
			),

			// Menu properties.
			'menu'        => array(
				'type'    => 'string',
				'format'  => 'menu_id',
				'example' => 'primary',
			),
			'mobile_toggle' => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),
			'submenu_click' => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),

			// Sidebar properties.
			'sidebar'     => array(
				'type'    => 'string',
				'format'  => 'sidebar_id',
				'example' => 'sidebar-1',
			),

			// Social properties.
			'platforms'   => array(
				'type'    => 'array',
				'format'  => 'social_platforms',
				'example' => array( 'facebook', 'twitter', 'instagram' ),
			),
			'target'      => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => '_blank',
			),
			'view'        => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'vertical',
			),

			// Map properties.
			'address'     => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '123 Main St, City, State',
			),

			// Counter properties.
			'number'      => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 100,
			),
			'max_number'  => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 1000,
			),
			'number_prefix' => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '$',
			),
			'number_suffix' => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '+',
			),

			// Callout properties.
			'cta_type'    => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'inline',
			),

			// Subscribe properties.
			'service'     => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'mailchimp',
			),

			// Slider properties.
			'slider_transition' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'slide',
			),
			'slider_speed' => array(
				'type'    => 'number',
				'format'  => 'milliseconds',
				'example' => 1000,
			),

			// Testimonials properties.
			'source'      => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'custom',
			),
		);

		// Return the schema for this property, or a generic schema if not found.
		if ( isset( $property_map[ $property ] ) ) {
			return $property_map[ $property ];
		}

		// Generic property schema.
		return array(
			'type'    => 'string',
			'format'  => 'text',
			'example' => 'value',
		);
	}

	/**
	 * Get structure notes for Beaver Builder.
	 *
	 * @since  1.4.0
	 * @return array Structure notes.
	 */
	private function get_structure_notes() {
		return array(
			'hierarchy'    => 'Row → Column → Module',
			'data_format'  => 'Beaver Builder stores data as serialized arrays in postmeta',
			'meta_key'     => '_fl_builder_data',
			'draft_key'    => '_fl_builder_draft',
			'modules'      => 'Modules are the content elements within columns',
			'note'         => 'Always work with the draft data when making changes, then publish',
		);
	}
}
