<?php
/**
 * Performance Analyzer
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/analyzers
 */

/**
 * Analyzes page performance, Core Web Vitals, and image optimization.
 *
 * @since 1.0.0
 */
class Respira_Performance_Analyzer {

	/**
	 * Analyze page performance metrics
	 *
	 * @since 1.0.0
	 * @param int $page_id Page ID to analyze.
	 * @return array Performance analysis results.
	 */
	public function analyze_performance( $page_id ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array(
				'success' => false,
				'message' => 'Page not found',
			);
		}

		$issues         = array();
		$recommendations = array();
		$metrics        = array();

		// Get post content and meta.
		$content    = $post->post_content;
		$post_meta  = get_post_meta( $page_id );

		// Analyze images.
		$image_analysis = $this->analyze_images_in_content( $content, $page_id );
		$metrics        = array_merge( $metrics, $image_analysis['metrics'] );
		$issues         = array_merge( $issues, $image_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $image_analysis['recommendations'] );

		// Analyze CSS/JS optimization.
		$script_analysis = $this->analyze_scripts( $page_id );
		$metrics        = array_merge( $metrics, $script_analysis['metrics'] );
		$issues         = array_merge( $issues, $script_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $script_analysis['recommendations'] );

		// Check caching status.
		$cache_analysis = $this->analyze_caching();
		$metrics       = array_merge( $metrics, $cache_analysis['metrics'] );
		$issues        = array_merge( $issues, $cache_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $cache_analysis['recommendations'] );

		// Check plugin performance impact.
		$plugin_analysis = $this->analyze_plugins();
		$metrics        = array_merge( $metrics, $plugin_analysis['metrics'] );
		$issues         = array_merge( $issues, $plugin_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $plugin_analysis['recommendations'] );

		// Calculate overall score (0-100).
		$score = $this->calculate_performance_score( $issues, $metrics );
		$grade = $this->get_grade_from_score( $score );

		return array(
			'success'         => true,
			'score'           => $score,
			'grade'           => $grade,
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Get Core Web Vitals metrics
	 *
	 * @since 1.0.0
	 * @param int $page_id Page ID to analyze.
	 * @return array Core Web Vitals analysis.
	 */
	public function get_core_web_vitals( $page_id ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array(
				'success' => false,
				'message' => 'Page not found',
			);
		}

		$issues         = array();
		$recommendations = array();
		$metrics        = array();

		// Estimate LCP (Largest Contentful Paint).
		$lcp_analysis   = $this->analyze_lcp( $page_id );
		$metrics['lcp'] = $lcp_analysis['estimated_lcp'];
		if ( $lcp_analysis['estimated_lcp'] > 2.5 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Estimated LCP exceeds recommended threshold',
				'severity' => 'high',
				'fix'      => 'Optimize largest content element (images, videos, or text blocks)',
			);
		}

		// Estimate FID (First Input Delay) - based on JavaScript presence.
		$fid_analysis   = $this->analyze_fid( $page_id );
		$metrics['fid'] = $fid_analysis['estimated_fid'];
		if ( $fid_analysis['estimated_fid'] > 100 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'High JavaScript execution time detected',
				'severity' => 'medium',
				'fix'      => 'Reduce JavaScript execution time and defer non-critical scripts',
			);
		}

		// Estimate CLS (Cumulative Layout Shift).
		$cls_analysis   = $this->analyze_cls( $page_id );
		$metrics['cls'] = $cls_analysis['estimated_cls'];
		if ( $cls_analysis['estimated_cls'] > 0.1 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Potential layout shift issues detected',
				'severity' => 'medium',
				'fix'      => 'Add explicit width/height to images and reserve space for dynamic content',
			);
		}

		// Calculate overall score.
		$score = $this->calculate_cwv_score( $metrics );
		$grade = $this->get_grade_from_score( $score );

		return array(
			'success'         => true,
			'score'           => $score,
			'grade'           => $grade,
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze images in page content
	 *
	 * @since 1.0.0
	 * @param int $page_id Page ID to analyze.
	 * @return array Image optimization analysis.
	 */
	public function analyze_images( $page_id ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array(
				'success' => false,
				'message' => 'Page not found',
			);
		}

		$content = $post->post_content;
		$analysis = $this->analyze_images_in_content( $content, $page_id );

		$score = $this->calculate_image_score( $analysis['issues'], $analysis['metrics'] );
		$grade = $this->get_grade_from_score( $score );

		return array(
			'success'         => true,
			'score'           => $score,
			'grade'           => $grade,
			'issues'          => $analysis['issues'],
			'recommendations' => $analysis['recommendations'],
			'metrics'         => $analysis['metrics'],
		);
	}

	/**
	 * Analyze images in content
	 *
	 * @param string $content Page content HTML.
	 * @param int    $page_id Page ID.
	 * @return array Image analysis.
	 */
	private function analyze_images_in_content( $content, $page_id ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'total_images'           => 0,
			'images_without_alt'     => 0,
			'large_images'           => 0,
			'unoptimized_formats'    => 0,
			'images_without_dimensions' => 0,
		);

		// Parse HTML to find images.
		preg_match_all( '/<img[^>]+>/i', $content, $matches );
		$images = $matches[0];

		$metrics['total_images'] = count( $images );

		foreach ( $images as $img_tag ) {
			// Check for alt text.
			if ( ! preg_match( '/alt=["\']([^"\']*)["\']/', $img_tag ) ) {
				$metrics['images_without_alt']++;
			}

			// Check for explicit dimensions.
			$has_width  = preg_match( '/width=["\']([^"\']*)["\']/', $img_tag );
			$has_height = preg_match( '/height=["\']([^"\']*)["\']/', $img_tag );
			if ( ! $has_width || ! $has_height ) {
				$metrics['images_without_dimensions']++;
			}

			// Check image URL for file size estimation.
			if ( preg_match( '/src=["\']([^"\']*)["\']/', $img_tag, $src_match ) ) {
				$img_url = $src_match[1];

				// Check if it's a local image.
				if ( strpos( $img_url, get_site_url() ) !== false ) {
					$img_path = str_replace( get_site_url(), ABSPATH, $img_url );
					$img_path = strtok( $img_path, '?' ); // Remove query string.

					if ( file_exists( $img_path ) ) {
						$file_size = filesize( $img_path );

						// Flag images over 500KB.
						if ( $file_size > 500000 ) {
							$metrics['large_images']++;
						}

						// Check for unoptimized formats (non-WebP).
						$extension = strtolower( pathinfo( $img_path, PATHINFO_EXTENSION ) );
						if ( in_array( $extension, array( 'jpg', 'jpeg', 'png' ), true ) ) {
							$metrics['unoptimized_formats']++;
						}
					}
				}
			}
		}

		// Generate issues based on metrics.
		if ( $metrics['images_without_alt'] > 0 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => sprintf( '%d images missing alt text', $metrics['images_without_alt'] ),
				'severity' => 'medium',
				'fix'      => 'Add descriptive alt text to all images for accessibility and SEO',
			);
		}

		if ( $metrics['large_images'] > 0 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => sprintf( '%d images exceed 500KB', $metrics['large_images'] ),
				'severity' => 'high',
				'fix'      => 'Compress images or use modern formats like WebP',
			);

			$recommendations[] = array(
				'priority' => 'high',
				'action'   => 'Optimize large images',
				'details'  => sprintf( '%d images exceed 500KB and should be compressed', $metrics['large_images'] ),
			);
		}

		if ( $metrics['unoptimized_formats'] > 0 ) {
			$recommendations[] = array(
				'priority' => 'medium',
				'action'   => 'Convert to modern image formats',
				'details'  => sprintf( '%d images could be converted to WebP for better compression', $metrics['unoptimized_formats'] ),
			);
		}

		if ( $metrics['images_without_dimensions'] > 0 ) {
			$issues[] = array(
				'type'     => 'info',
				'message'  => sprintf( '%d images missing explicit dimensions', $metrics['images_without_dimensions'] ),
				'severity' => 'low',
				'fix'      => 'Add width and height attributes to prevent layout shifts',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze scripts and styles
	 *
	 * @param int $page_id Page ID.
	 * @return array Script analysis.
	 */
	private function analyze_scripts( $page_id ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'total_scripts'      => 0,
			'total_styles'       => 0,
			'inline_scripts'     => 0,
			'inline_styles'      => 0,
			'render_blocking'    => 0,
		);

		// Get enqueued scripts and styles.
		global $wp_scripts, $wp_styles;

		if ( ! empty( $wp_scripts ) && is_object( $wp_scripts ) ) {
			$metrics['total_scripts'] = count( $wp_scripts->queue );
		}

		if ( ! empty( $wp_styles ) && is_object( $wp_styles ) ) {
			$metrics['total_styles'] = count( $wp_styles->queue );
		}

		// Check for excessive scripts/styles.
		if ( $metrics['total_scripts'] > 15 ) {
			$recommendations[] = array(
				'priority' => 'medium',
				'action'   => 'Reduce JavaScript files',
				'details'  => sprintf( '%d JavaScript files detected. Consider combining or deferring non-critical scripts', $metrics['total_scripts'] ),
			);
		}

		if ( $metrics['total_styles'] > 10 ) {
			$recommendations[] = array(
				'priority' => 'medium',
				'action'   => 'Reduce CSS files',
				'details'  => sprintf( '%d CSS files detected. Consider combining stylesheets', $metrics['total_styles'] ),
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze caching configuration
	 *
	 * @return array Cache analysis.
	 */
	private function analyze_caching() {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'object_cache_enabled' => false,
			'page_cache_enabled'   => false,
			'cache_plugin_active'  => false,
		);

		// Check for object cache.
		$metrics['object_cache_enabled'] = wp_using_ext_object_cache();

		// Check for common caching plugins.
		$cache_plugins = array(
			'wp-super-cache/wp-cache.php',
			'w3-total-cache/w3-total-cache.php',
			'wp-fastest-cache/wpFastestCache.php',
			'wp-rocket/wp-rocket.php',
			'litespeed-cache/litespeed-cache.php',
		);

		foreach ( $cache_plugins as $plugin ) {
			if ( is_plugin_active( $plugin ) ) {
				$metrics['cache_plugin_active'] = true;
				$metrics['page_cache_enabled']  = true;
				break;
			}
		}

		if ( ! $metrics['page_cache_enabled'] ) {
			$recommendations[] = array(
				'priority' => 'high',
				'action'   => 'Enable page caching',
				'details'  => 'Install a caching plugin like WP Rocket, W3 Total Cache, or LiteSpeed Cache',
			);
		}

		if ( ! $metrics['object_cache_enabled'] ) {
			$recommendations[] = array(
				'priority' => 'medium',
				'action'   => 'Enable object caching',
				'details'  => 'Configure Redis or Memcached for improved database performance',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze plugin performance impact
	 *
	 * @return array Plugin analysis.
	 */
	private function analyze_plugins() {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'total_plugins'      => 0,
			'active_plugins'     => 0,
		);

		$all_plugins    = get_plugins();
		$active_plugins = get_option( 'active_plugins', array() );

		$metrics['total_plugins']  = count( $all_plugins );
		$metrics['active_plugins'] = count( $active_plugins );

		if ( $metrics['active_plugins'] > 30 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => sprintf( '%d active plugins detected', $metrics['active_plugins'] ),
				'severity' => 'medium',
				'fix'      => 'Review and deactivate unused plugins to improve performance',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze LCP (Largest Contentful Paint)
	 *
	 * @param int $page_id Page ID.
	 * @return array LCP analysis.
	 */
	private function analyze_lcp( $page_id ) {
		// This is an estimation based on content analysis.
		// Real LCP requires field data or lab testing.
		$post    = get_post( $page_id );
		$content = $post->post_content;

		// Estimate based on content complexity.
		$image_count = substr_count( $content, '<img' );
		$video_count = substr_count( $content, '<video' );

		// Simple heuristic: more media = longer LCP.
		$estimated_lcp = 1.5 + ( $image_count * 0.2 ) + ( $video_count * 0.5 );

		return array(
			'estimated_lcp' => $estimated_lcp,
		);
	}

	/**
	 * Analyze FID (First Input Delay)
	 *
	 * @param int $page_id Page ID.
	 * @return array FID analysis.
	 */
	private function analyze_fid( $page_id ) {
		// This is an estimation based on JavaScript complexity.
		global $wp_scripts;

		$script_count = 0;
		if ( ! empty( $wp_scripts ) && is_object( $wp_scripts ) ) {
			$script_count = count( $wp_scripts->queue );
		}

		// Simple heuristic: more scripts = longer FID.
		$estimated_fid = 50 + ( $script_count * 10 );

		return array(
			'estimated_fid' => $estimated_fid,
		);
	}

	/**
	 * Analyze CLS (Cumulative Layout Shift)
	 *
	 * @param int $page_id Page ID.
	 * @return array CLS analysis.
	 */
	private function analyze_cls( $page_id ) {
		// This is an estimation based on content structure.
		$post    = get_post( $page_id );
		$content = $post->post_content;

		// Check for images without dimensions.
		$images_without_dims = 0;
		preg_match_all( '/<img[^>]+>/i', $content, $matches );
		foreach ( $matches[0] as $img_tag ) {
			$has_width  = preg_match( '/width=["\']([^"\']*)["\']/', $img_tag );
			$has_height = preg_match( '/height=["\']([^"\']*)["\']/', $img_tag );
			if ( ! $has_width || ! $has_height ) {
				$images_without_dims++;
			}
		}

		// Simple heuristic: images without dimensions cause layout shift.
		$estimated_cls = $images_without_dims * 0.05;

		return array(
			'estimated_cls' => $estimated_cls,
		);
	}

	/**
	 * Calculate performance score
	 *
	 * @param array $issues Issues found.
	 * @param array $metrics Metrics collected.
	 * @return int Score (0-100).
	 */
	private function calculate_performance_score( $issues, $metrics ) {
		$score = 100;

		// Deduct points for issues.
		foreach ( $issues as $issue ) {
			switch ( $issue['severity'] ) {
				case 'high':
					$score -= 15;
					break;
				case 'medium':
					$score -= 10;
					break;
				case 'low':
					$score -= 5;
					break;
			}
		}

		// Deduct points for poor metrics.
		if ( isset( $metrics['large_images'] ) && $metrics['large_images'] > 3 ) {
			$score -= 10;
		}

		if ( isset( $metrics['active_plugins'] ) && $metrics['active_plugins'] > 30 ) {
			$score -= 10;
		}

		return max( 0, $score );
	}

	/**
	 * Calculate Core Web Vitals score
	 *
	 * @param array $metrics Metrics collected.
	 * @return int Score (0-100).
	 */
	private function calculate_cwv_score( $metrics ) {
		$score = 100;

		// LCP scoring.
		if ( isset( $metrics['lcp'] ) ) {
			if ( $metrics['lcp'] > 4.0 ) {
				$score -= 30;
			} elseif ( $metrics['lcp'] > 2.5 ) {
				$score -= 15;
			}
		}

		// FID scoring.
		if ( isset( $metrics['fid'] ) ) {
			if ( $metrics['fid'] > 300 ) {
				$score -= 30;
			} elseif ( $metrics['fid'] > 100 ) {
				$score -= 15;
			}
		}

		// CLS scoring.
		if ( isset( $metrics['cls'] ) ) {
			if ( $metrics['cls'] > 0.25 ) {
				$score -= 30;
			} elseif ( $metrics['cls'] > 0.1 ) {
				$score -= 15;
			}
		}

		return max( 0, $score );
	}

	/**
	 * Calculate image optimization score
	 *
	 * @param array $issues Issues found.
	 * @param array $metrics Metrics collected.
	 * @return int Score (0-100).
	 */
	private function calculate_image_score( $issues, $metrics ) {
		$score = 100;

		// Deduct points for issues.
		foreach ( $issues as $issue ) {
			switch ( $issue['severity'] ) {
				case 'high':
					$score -= 20;
					break;
				case 'medium':
					$score -= 15;
					break;
				case 'low':
					$score -= 5;
					break;
			}
		}

		// Deduct points based on metrics.
		if ( isset( $metrics['total_images'] ) && $metrics['total_images'] > 0 ) {
			$alt_coverage = 1 - ( $metrics['images_without_alt'] / $metrics['total_images'] );
			if ( $alt_coverage < 0.5 ) {
				$score -= 20;
			} elseif ( $alt_coverage < 0.8 ) {
				$score -= 10;
			}
		}

		return max( 0, $score );
	}

	/**
	 * Get letter grade from score
	 *
	 * @param int $score Score (0-100).
	 * @return string Letter grade.
	 */
	private function get_grade_from_score( $score ) {
		if ( $score >= 90 ) {
			return 'A';
		} elseif ( $score >= 80 ) {
			return 'B';
		} elseif ( $score >= 70 ) {
			return 'C';
		} elseif ( $score >= 60 ) {
			return 'D';
		} else {
			return 'F';
		}
	}
}
