<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * @since 1.0.0
 */
class Respira_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 * @var    string $version The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_styles() {
		$css_file = RESPIRA_PLUGIN_DIR . 'admin/assets/css/respira-admin.css';
		$version  = file_exists( $css_file ) ? filemtime( $css_file ) : $this->version;

		// Enqueue with high priority to override theme styles
		wp_enqueue_style(
			$this->plugin_name,
			RESPIRA_PLUGIN_URL . 'admin/assets/css/respira-admin.css',
			array(),
			$version,
			'all'
		);

		// Add inline critical styles to ensure they override theme styles
		$critical_css = '
			body[class*="respira-for-wordpress"] #wpwrap,
			body[class*="respira-for-wordpress"] #wpbody,
			body[class*="respira-for-wordpress"] #wpcontent,
			body[class*="respira-for-wordpress"] #wpbody-content {
				background: #F1FAEE !important;
			}
			body[class*="respira-for-wordpress"] .wrap,
			body[class*="respira-for-wordpress"] #wpbody-content > .wrap {
				background: transparent !important;
			}
		';
		wp_add_inline_style( $this->plugin_name, $critical_css );
	}

	/**
	 * Register AJAX handlers (called on admin_init).
	 *
	 * @since 1.5.1
	 */
	public function register_ajax_handlers() {
		// Register AJAX handlers - must be registered early, not in enqueue.
		add_action( 'wp_ajax_respira_approve_duplicate', array( $this, 'ajax_approve_duplicate' ) );
		add_action( 'wp_ajax_respira_reject_duplicate', array( $this, 'ajax_reject_duplicate' ) );
		add_action( 'wp_ajax_respira_submit_testimonial', array( $this, 'ajax_submit_testimonial' ) );
		add_action( 'wp_ajax_respira_update_plugin', array( $this, 'ajax_update_plugin' ) );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script(
			$this->plugin_name,
			RESPIRA_PLUGIN_URL . 'admin/assets/js/respira-admin.js',
			array( 'jquery' ),
			$this->version,
			false
		);

		// Localize script for AJAX.
		wp_localize_script(
			$this->plugin_name,
			'respira_admin',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'respira_admin_nonce' ),
			)
		);
	}

	/**
	 * Add plugin admin menu.
	 *
	 * @since 1.0.0
	 */
	public function add_plugin_admin_menu() {
		// Use official 20x20 SVG icon per WordPress admin menu guidelines.
		$icon_url = RESPIRA_PLUGIN_URL . 'admin/assets/images/icon.svg';
		
		// Main menu page - uses dashboard as callback.
		add_menu_page(
			__( 'Respira for WordPress', 'respira-for-wordpress' ),
			__( 'Respira', 'respira-for-wordpress' ),
			'manage_options',
			'respira-for-wordpress',
			array( $this, 'display_dashboard_page' ),
			$icon_url,
			30
		);
		

		// Dashboard - override the auto-generated submenu item.
		add_submenu_page(
			'respira-for-wordpress',
			__( 'Dashboard', 'respira-for-wordpress' ),
			__( 'Dashboard', 'respira-for-wordpress' ),
			'manage_options',
			'respira-for-wordpress',
			array( $this, 'display_dashboard_page' )
		);

		// License (before API Keys).
		add_submenu_page(
			'respira-for-wordpress',
			__( 'License', 'respira-for-wordpress' ),
			__( 'License', 'respira-for-wordpress' ),
			'manage_options',
			'respira-license',
			array( $this, 'display_license_page' )
		);

		// Start Here.
		add_submenu_page(
			'respira-for-wordpress',
			__( 'Start Here', 'respira-for-wordpress' ),
			__( 'Start Here', 'respira-for-wordpress' ),
			'manage_options',
			'respira-start-here',
			array( $this, 'display_start_here_page' )
		);

		// API Keys.
		add_submenu_page(
			'respira-for-wordpress',
			__( 'API Keys', 'respira-for-wordpress' ),
			__( 'API Keys', 'respira-for-wordpress' ),
			'manage_options',
			'respira-api-keys',
			array( $this, 'display_api_keys_page' )
		);

		// Settings.
		add_submenu_page(
			'respira-for-wordpress',
			__( 'Settings', 'respira-for-wordpress' ),
			__( 'Settings', 'respira-for-wordpress' ),
			'manage_options',
			'respira-settings',
			array( $this, 'display_settings_page' )
		);

		// Approve Edits.
		add_submenu_page(
			'respira-for-wordpress',
			__( 'Approve Edits', 'respira-for-wordpress' ),
			__( 'Approve Edits', 'respira-for-wordpress' ),
			'manage_options',
			'respira-approvals',
			array( $this, 'display_approvals_page' )
		);

		// Audit Log.
		add_submenu_page(
			'respira-for-wordpress',
			__( 'Audit Log', 'respira-for-wordpress' ),
			__( 'Audit Log', 'respira-for-wordpress' ),
			'manage_options',
			'respira-audit-log',
			array( $this, 'display_audit_log_page' )
		);

		// Remove the duplicate submenu item that WordPress auto-creates.
		global $submenu;
		if ( isset( $submenu['respira-for-wordpress'] ) ) {
			// Remove the first item if it's a duplicate (same slug as parent).
			foreach ( $submenu['respira-for-wordpress'] as $key => $item ) {
				if ( 'respira-for-wordpress' === $item[2] && 0 !== $key ) {
					unset( $submenu['respira-for-wordpress'][ $key ] );
					break;
				}
			}
		}
	}

	/**
	 * Display license page.
	 *
	 * @since 1.5.3
	 */
	public function display_license_page() {
		// Handle license activation.
		if ( isset( $_POST['respira_activate_license'] ) && check_admin_referer( 'respira_activate_license' ) ) {
			$license_key = sanitize_text_field( $_POST['license_key'] ?? '' );
			$agree_privacy = isset( $_POST['agree_privacy'] ) && '1' === $_POST['agree_privacy'];
			$agree_terms = isset( $_POST['agree_terms'] ) && '1' === $_POST['agree_terms'];
			
			// Validate that both checkboxes are checked
			if ( ! $agree_privacy || ! $agree_terms ) {
				add_settings_error( 'respira_messages', 'respira_error', __( 'You must confirm that you have read and agree to both the Privacy Policy and Terms of Service before activating your license.', 'respira-for-wordpress' ), 'error' );
			} elseif ( empty( $license_key ) ) {
				add_settings_error( 'respira_messages', 'respira_error', __( 'Please enter a license key.', 'respira-for-wordpress' ), 'error' );
			} else {
				$result = Respira_License::activate_license( $license_key );
				
				if ( is_wp_error( $result ) ) {
					add_settings_error( 'respira_messages', 'respira_error', $result->get_error_message(), 'error' );
				} else {
					add_settings_error( 'respira_messages', 'respira_success', __( 'License activated successfully!', 'respira-for-wordpress' ), 'success' );
				}
			}
		}

		// Handle license deactivation.
		if ( isset( $_POST['respira_deactivate_license'] ) && check_admin_referer( 'respira_deactivate_license' ) ) {
			$result = Respira_License::deactivate_license();
			
			if ( is_wp_error( $result ) ) {
				add_settings_error( 'respira_messages', 'respira_error', $result->get_error_message(), 'error' );
			} else {
				add_settings_error( 'respira_messages', 'respira_success', __( 'License deactivated successfully.', 'respira-for-wordpress' ), 'success' );
			}
		}

		require_once RESPIRA_PLUGIN_DIR . 'admin/views/license.php';
	}

	/**
	 * Display start here page.
	 *
	 * @since 1.1.1
	 */
	public function display_start_here_page() {
		require_once RESPIRA_PLUGIN_DIR . 'admin/views/start-here.php';
	}

	/**
	 * Display dashboard page.
	 *
	 * @since 1.0.0
	 */
	public function display_dashboard_page() {
		require_once RESPIRA_PLUGIN_DIR . 'admin/views/dashboard.php';
	}

	/**
	 * Display API keys page.
	 *
	 * @since 1.0.0
	 */
	public function display_api_keys_page() {
		// Check if license is active first.
		$license_status = get_option( 'respira_license_status', 'inactive' );
		if ( 'active' !== $license_status && 'trial' !== $license_status ) {
			add_settings_error( 'respira_messages', 'respira_error', __( 'Please activate your license first before generating API keys.', 'respira-for-wordpress' ), 'error' );
		}

		// Handle form submissions.
		if ( isset( $_POST['respira_generate_key'] ) && check_admin_referer( 'respira_generate_key' ) ) {
			$name    = sanitize_text_field( $_POST['key_name'] ?? '' );
			$api_key = Respira_Auth::generate_api_key( get_current_user_id(), $name );

			if ( is_wp_error( $api_key ) ) {
				add_settings_error( 'respira_messages', 'respira_error', $api_key->get_error_message(), 'error' );
			} else {
				// Store in transient to show once.
				set_transient( 'respira_new_api_key', $api_key, 60 );
				add_settings_error( 'respira_messages', 'respira_success', __( 'API key generated successfully!', 'respira-for-wordpress' ), 'success' );
			}
		}

		if ( isset( $_POST['respira_revoke_key'] ) && check_admin_referer( 'respira_revoke_key' ) ) {
			$key_id = absint( $_POST['key_id'] );
			$result = Respira_Auth::revoke_api_key( $key_id );

			if ( is_wp_error( $result ) ) {
				add_settings_error( 'respira_messages', 'respira_error', $result->get_error_message(), 'error' );
			} else {
				add_settings_error( 'respira_messages', 'respira_success', __( 'API key revoked successfully!', 'respira-for-wordpress' ), 'success' );
			}
		}

		require_once RESPIRA_PLUGIN_DIR . 'admin/views/api-keys.php';
	}

	/**
	 * Display settings page.
	 *
	 * @since 1.0.0
	 */
	public function display_settings_page() {
		// Handle settings update.
		if ( isset( $_POST['respira_update_settings'] ) && check_admin_referer( 'respira_update_settings' ) ) {
			update_option( 'respira_auto_duplicate', isset( $_POST['auto_duplicate'] ) ? 1 : 0 );
			update_option( 'respira_security_checks', isset( $_POST['security_checks'] ) ? 1 : 0 );
			update_option( 'respira_audit_logging', isset( $_POST['audit_logging'] ) ? 1 : 0 );
			update_option( 'respira_allow_direct_edit', isset( $_POST['allow_direct_edit'] ) ? 1 : 0 );
			update_option( 'respira_enable_plugin_management', isset( $_POST['enable_plugin_management'] ) ? 1 : 0 );
			update_option( 'respira_rate_limit', absint( $_POST['rate_limit'] ?? 100 ) );

			add_settings_error( 'respira_messages', 'respira_success', __( 'Settings saved successfully!', 'respira-for-wordpress' ), 'success' );
		}

		require_once RESPIRA_PLUGIN_DIR . 'admin/views/settings.php';
	}

	/**
	 * Display approvals page.
	 *
	 * @since 1.2.0
	 */
	public function display_approvals_page() {
		require_once RESPIRA_PLUGIN_DIR . 'admin/views/approvals.php';
	}

	/**
	 * Display audit log page.
	 *
	 * @since 1.0.0
	 */
	public function display_audit_log_page() {
		require_once RESPIRA_PLUGIN_DIR . 'admin/views/audit-log.php';
	}

	/**
	 * AJAX handler for approving duplicates.
	 *
	 * @since 1.2.0
	 */
	public function ajax_approve_duplicate() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'respira_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'respira-for-wordpress' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'respira-for-wordpress' ) ) );
		}

		// Get duplicate ID.
		$duplicate_id = isset( $_POST['duplicate_id'] ) ? absint( $_POST['duplicate_id'] ) : 0;
		if ( ! $duplicate_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid duplicate ID.', 'respira-for-wordpress' ) ) );
		}

		// Approve the duplicate.
		$result = Respira_Approver::approve_duplicate( $duplicate_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success(
			array(
				'message' => __( 'Duplicate approved successfully. The original has been unpublished and the duplicate is now live.', 'respira-for-wordpress' ),
			)
		);
	}

	/**
	 * AJAX handler for rejecting duplicates.
	 *
	 * @since 1.4.2
	 */
	public function ajax_reject_duplicate() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'respira_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'respira-for-wordpress' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'respira-for-wordpress' ) ) );
		}

		// Get duplicate ID.
		$duplicate_id = isset( $_POST['duplicate_id'] ) ? absint( $_POST['duplicate_id'] ) : 0;
		if ( ! $duplicate_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid duplicate ID.', 'respira-for-wordpress' ) ) );
		}

		// Reject the duplicate.
		$result = Respira_Approver::reject_duplicate( $duplicate_id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		wp_send_json_success(
			array(
				'message' => __( 'Duplicate rejected and deleted successfully.', 'respira-for-wordpress' ),
			)
		);
	}

	/**
	 * AJAX handler for submitting testimonials.
	 *
	 * @since 1.8.0
	 */
	public function ajax_submit_testimonial() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'respira_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'respira-for-wordpress' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'respira-for-wordpress' ) ) );
		}

		// Get form data.
		$name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
		$testimonial = isset( $_POST['testimonial'] ) ? sanitize_textarea_field( wp_unslash( $_POST['testimonial'] ) ) : '';
		$rating = isset( $_POST['rating'] ) ? absint( $_POST['rating'] ) : 0;
		$anonymous_name = isset( $_POST['anonymous_name'] ) && '1' === $_POST['anonymous_name'];
		$anonymous_website = isset( $_POST['anonymous_website'] ) && '1' === $_POST['anonymous_website'];
		$website_url = isset( $_POST['website_url'] ) ? esc_url_raw( wp_unslash( $_POST['website_url'] ) ) : '';

		// Validate required fields.
		if ( empty( $name ) || empty( $testimonial ) || empty( $rating ) || $rating < 1 || $rating > 5 ) {
			wp_send_json_error( array( 'message' => __( 'Please fill in all required fields.', 'respira-for-wordpress' ) ) );
		}

		// Submit testimonial to Supabase via API
		$api_url = 'https://respira.press/api/reviews/submit-from-plugin';
		$response = wp_remote_post(
			$api_url,
			array(
				'timeout' => 15,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode(
					array(
						'name'             => $name,
						'testimonial'      => $testimonial,
						'rating'           => $rating,
						'anonymous_name'   => $anonymous_name ? '1' : '0',
						'anonymous_website' => $anonymous_website ? '1' : '0',
						'website_url'      => $website_url,
					)
				),
			)
		);

		// Check for API errors
		if ( is_wp_error( $response ) ) {
			// API call failed - fallback to local storage
			$testimonials = get_option( 'respira_testimonials_pending', array() );
			$testimonials[] = array(
				'name'            => $name,
				'testimonial'     => $testimonial,
				'rating'          => $rating,
				'anonymous_name'  => $anonymous_name,
				'anonymous_website' => $anonymous_website,
				'website_url'     => $website_url,
				'submitted_at'    => current_time( 'mysql' ),
				'submitted_by'    => get_current_user_id(),
				'sync_error'      => $response->get_error_message(),
			);
			update_option( 'respira_testimonials_pending', $testimonials );

			wp_send_json_success(
				array(
					'message' => __( 'Testimonial saved locally. There was an issue syncing to our server, but it will be reviewed.', 'respira-for-wordpress' ),
				)
			);
			return;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $response_body, true );

		if ( 200 !== $response_code || ! isset( $response_data['success'] ) || ! $response_data['success'] ) {
			// API call failed - fallback to local storage
			$error_message = isset( $response_data['error'] ) ? $response_data['error'] : __( 'Unknown error', 'respira-for-wordpress' );
			$testimonials = get_option( 'respira_testimonials_pending', array() );
			$testimonials[] = array(
				'name'            => $name,
				'testimonial'     => $testimonial,
				'rating'          => $rating,
				'anonymous_name'  => $anonymous_name,
				'anonymous_website' => $anonymous_website,
				'website_url'     => $website_url,
				'submitted_at'    => current_time( 'mysql' ),
				'submitted_by'    => get_current_user_id(),
				'sync_error'      => $error_message,
			);
			update_option( 'respira_testimonials_pending', $testimonials );

			wp_send_json_success(
				array(
					'message' => __( 'Testimonial saved locally. There was an issue syncing to our server, but it will be reviewed.', 'respira-for-wordpress' ),
				)
			);
			return;
		}

		// Successfully submitted to Supabase
		wp_send_json_success(
			array(
				'message' => __( 'Testimonial submitted successfully! It will be reviewed by our team.', 'respira-for-wordpress' ),
			)
		);
	}

	/**
	 * AJAX handler for updating the plugin.
	 *
	 * @since 1.8.7
	 */
	public function ajax_update_plugin() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'respira_admin_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'respira-for-wordpress' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'respira-for-wordpress' ) ) );
		}

		// Check if update is available.
		$update_info = Respira_Updater::check_update_available();
		if ( ! $update_info || ! $update_info['available'] ) {
			wp_send_json_error( array( 'message' => __( 'No update available.', 'respira-for-wordpress' ) ) );
		}

		// Include required WordPress files.
		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		// Get the plugin basename.
		$plugin_basename = RESPIRA_PLUGIN_BASENAME;

		// Get download URL from update server.
		$updater = new Respira_Updater();
		$release_info = $updater->get_latest_release();
		
		if ( is_wp_error( $release_info ) || ! isset( $release_info['download_url'] ) ) {
			wp_send_json_error( array( 'message' => __( 'Failed to get update package URL.', 'respira-for-wordpress' ) ) );
		}

		$download_url = $release_info['download_url'];
		
		// If it's a GitHub URL, use proxy endpoint.
		if ( strpos( $download_url, 'github.com/repos' ) !== false || strpos( $download_url, 'githubusercontent.com' ) !== false ) {
			$download_url = add_query_arg(
				array( 'version' => $release_info['version'] ),
				'https://respira.press/api/plugin-updates/download'
			);
		}

		// Create a minimal upgrader skin for AJAX operations.
		if ( ! class_exists( 'WP_Ajax_Upgrader_Skin' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php';
		}
		$skin = new WP_Ajax_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader( $skin );

		// Perform the update using the package URL.
		$result = $upgrader->run(
			array(
				'package'                     => $download_url,
				'destination'                 => WP_PLUGIN_DIR,
				'clear_destination'           => true,
				'clear_working'               => true,
				'hook_extra'                  => array(
					'plugin' => $plugin_basename,
					'type'   => 'plugin',
					'action' => 'update',
				),
			)
		);

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}

		if ( false === $result ) {
			$errors = $skin->get_errors();
			$error_message = ! empty( $errors ) ? implode( ' ', $errors ) : __( 'Update failed. Please try again or update manually.', 'respira-for-wordpress' );
			wp_send_json_error( array( 'message' => $error_message ) );
		}

		// Clear update cache.
		Respira_Updater::clear_cache();

		// Activate the plugin if it was active before.
		if ( is_plugin_active( $plugin_basename ) ) {
			activate_plugin( $plugin_basename );
		}

		wp_send_json_success( array( 'message' => __( 'Plugin updated successfully!', 'respira-for-wordpress' ) ) );
	}
}
